/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import com.google.common.collect.Ordering;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogArchiver;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogPosition;
import org.apache.cassandra.db.commitlog.CommitLogReadHandler;
import org.apache.cassandra.db.commitlog.CommitLogReader;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.db.commitlog.CommitLogSegmentManagerCDC;
import org.apache.cassandra.db.commitlog.IntervalSet;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.schema.Schema;
import org.apache.cassandra.schema.TableId;
import org.apache.cassandra.schema.TableMetadataRef;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.WrappedRunnable;
import org.apache.cassandra.utils.concurrent.Future;
import org.apache.commons.lang3.StringUtils;
import org.cliffc.high_scale_lib.NonBlockingHashSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogReplayer
implements CommitLogReadHandler {
    @VisibleForTesting
    public static long MAX_OUTSTANDING_REPLAY_BYTES = CassandraRelevantProperties.COMMITLOG_MAX_OUTSTANDING_REPLAY_BYTES.getLong();
    @VisibleForTesting
    public static MutationInitiator mutationInitiator = new MutationInitiator();
    private static final Logger logger = LoggerFactory.getLogger(CommitLogReplayer.class);
    private static final int MAX_OUTSTANDING_REPLAY_COUNT = CassandraRelevantProperties.COMMITLOG_MAX_OUTSTANDING_REPLAY_COUNT.getInt();
    private final Set<Keyspace> keyspacesReplayed = new NonBlockingHashSet<Keyspace>();
    private final Queue<Future<Integer>> futures = new ArrayDeque<Future<Integer>>();
    private final AtomicInteger replayedCount = new AtomicInteger();
    private final Map<TableId, IntervalSet<CommitLogPosition>> cfPersisted;
    private final CommitLogPosition globalPosition;
    private long pendingMutationBytes = 0L;
    private final ReplayFilter replayFilter;
    private CommitLogArchiver archiver;
    @VisibleForTesting
    protected boolean sawCDCMutation;
    @VisibleForTesting
    protected CommitLogReader commitLogReader;

    CommitLogReplayer(CommitLog commitLog, CommitLogPosition globalPosition, Map<TableId, IntervalSet<CommitLogPosition>> cfPersisted, ReplayFilter replayFilter) {
        this.cfPersisted = cfPersisted;
        this.globalPosition = globalPosition;
        this.replayFilter = replayFilter;
        this.archiver = commitLog.archiver;
        this.commitLogReader = new CommitLogReader();
    }

    public static CommitLogReplayer construct(CommitLog commitLog, UUID localHostId) {
        HashMap<TableId, IntervalSet<CommitLogPosition>> cfPersisted = new HashMap<TableId, IntervalSet<CommitLogPosition>>();
        ReplayFilter replayFilter = ReplayFilter.create();
        for (ColumnFamilyStore cfs : ColumnFamilyStore.all()) {
            IntervalSet<CommitLogPosition> filter;
            CommitLogPosition snapshotPosition;
            CommitLogPosition truncatedAt = SystemKeyspace.getTruncatedPosition(cfs.metadata.id);
            if (truncatedAt != null) {
                long restoreTime = commitLog.archiver.restorePointInTimeInMicroseconds == Long.MAX_VALUE ? Long.MAX_VALUE : commitLog.archiver.restorePointInTimeInMicroseconds / 1000L;
                long truncatedTime = SystemKeyspace.getTruncatedAt(cfs.metadata.id);
                if (truncatedTime > restoreTime && replayFilter.includes(cfs.metadata)) {
                    logger.info("Restore point in time is before latest truncation of table {}.{}. Clearing truncation record.", (Object)cfs.metadata.keyspace, (Object)cfs.metadata.name);
                    SystemKeyspace.removeTruncationRecord(cfs.metadata.id);
                    truncatedAt = null;
                }
            }
            if ((snapshotPosition = commitLog.archiver.snapshotCommitLogPosition) == CommitLogPosition.NONE) {
                if (!cfs.memtableWritesAreDurable()) {
                    filter = CommitLogReplayer.persistedIntervals(cfs.getLiveSSTables(), truncatedAt, localHostId);
                } else if (commitLog.archiver.getRestorePointInTimeInMicroseconds() == Long.MAX_VALUE) {
                    filter = new IntervalSet<CommitLogPosition>(CommitLogPosition.NONE, CommitLog.instance.getCurrentPosition());
                } else {
                    logger.info("Point-in-time restore on a persistent memtable started without a snapshot time. All commit log data will be replayed.");
                    filter = IntervalSet.empty();
                }
            } else {
                filter = new IntervalSet<CommitLogPosition>(CommitLogPosition.NONE, snapshotPosition);
            }
            cfPersisted.put(cfs.metadata.id, filter);
        }
        CommitLogPosition globalPosition = CommitLogReplayer.firstNotCovered(cfPersisted.values());
        logger.debug("Global replay position is {} from columnfamilies {}", (Object)globalPosition, (Object)FBUtilities.toString(cfPersisted));
        return new CommitLogReplayer(commitLog, globalPosition, cfPersisted, replayFilter);
    }

    public void replayPath(File file, boolean tolerateTruncation) throws IOException {
        this.sawCDCMutation = false;
        this.commitLogReader.readCommitLogSegment(this, file, this.globalPosition, -1, tolerateTruncation);
        if (this.sawCDCMutation) {
            this.handleCDCReplayCompletion(file);
        }
    }

    public void replayFiles(File[] clogs) throws IOException {
        List<File> filteredLogs = CommitLogReader.filterCommitLogFiles(clogs);
        int i = 0;
        for (File file : filteredLogs) {
            this.sawCDCMutation = false;
            this.commitLogReader.readCommitLogSegment((CommitLogReadHandler)this, file, this.globalPosition, ++i == filteredLogs.size());
            if (!this.sawCDCMutation) continue;
            this.handleCDCReplayCompletion(file);
        }
    }

    private void handleCDCReplayCompletion(File f) throws IOException {
        ((CommitLogSegmentManagerCDC)CommitLog.instance.segmentManager).addCDCSize(f.length());
        File dest = new File(DatabaseDescriptor.getCDCLogLocation(), f.name());
        if (!dest.exists()) {
            FileUtils.createHardLink(f, dest);
        }
        try (RandomAccessReader reader = RandomAccessReader.open(f);){
            CommitLogDescriptor desc = CommitLogDescriptor.readHeader(reader, DatabaseDescriptor.getEncryptionContext());
            assert (desc != null);
            assert (f.length() < Integer.MAX_VALUE);
            CommitLogSegment.writeCDCIndexFile(desc, (int)f.length(), true);
        }
    }

    public int blockForWrites() {
        for (Map.Entry<TableId, AtomicInteger> entry : this.commitLogReader.getInvalidMutations()) {
            logger.warn("Skipped {} mutations from unknown (probably removed) CF with id {}", (Object)entry.getValue(), (Object)entry.getKey());
        }
        FBUtilities.waitOnFutures(this.futures);
        logger.trace("Finished waiting on mutations from recovery");
        this.futures.clear();
        boolean flushingSystem = false;
        ArrayList futures = new ArrayList();
        for (Keyspace keyspace : this.keyspacesReplayed) {
            if (keyspace.getName().equals("system")) {
                flushingSystem = true;
            }
            futures.addAll(keyspace.flush(ColumnFamilyStore.FlushReason.STARTUP));
        }
        if (!flushingSystem) {
            futures.add(Keyspace.open("system").getColumnFamilyStore("batches").forceFlush(ColumnFamilyStore.FlushReason.INTERNALLY_FORCED));
        }
        FBUtilities.waitOnFutures(futures);
        return this.replayedCount.get();
    }

    public static IntervalSet<CommitLogPosition> persistedIntervals(Iterable<SSTableReader> onDisk, CommitLogPosition truncatedAt, UUID localhostId) {
        IntervalSet.Builder<CommitLogPosition> builder = new IntervalSet.Builder<CommitLogPosition>();
        ArrayList<String> skippedSSTables = new ArrayList<String>();
        for (SSTableReader reader : onDisk) {
            UUID originatingHostId = reader.getSSTableMetadata().originatingHostId;
            if (originatingHostId != null && originatingHostId.equals(localhostId)) {
                builder.addAll(reader.getSSTableMetadata().commitLogIntervals);
                continue;
            }
            skippedSSTables.add(reader.getFilename());
        }
        if (!skippedSSTables.isEmpty()) {
            logger.warn("Origin of {} sstables is unknown or doesn't match the local node; commitLogIntervals for them were ignored", (Object)skippedSSTables.size());
            logger.debug("Ignored commitLogIntervals from the following sstables: {}", skippedSSTables);
        }
        if (truncatedAt != null) {
            builder.add(CommitLogPosition.NONE, truncatedAt);
        }
        return builder.build();
    }

    public static CommitLogPosition firstNotCovered(Collection<IntervalSet<CommitLogPosition>> ranges) {
        return ranges.stream().map(intervals -> Iterables.getFirst(intervals.ends(), CommitLogPosition.NONE)).min(Ordering.natural()).get();
    }

    private boolean shouldReplay(TableId tableId, CommitLogPosition position) {
        return !this.cfPersisted.get(tableId).contains(position);
    }

    protected boolean pointInTimeExceeded(Mutation fm) {
        for (PartitionUpdate upd : fm.getPartitionUpdates()) {
            if (this.archiver.precision.toMicros(upd.maxTimestamp()) <= this.archiver.restorePointInTimeInMicroseconds) continue;
            return true;
        }
        return false;
    }

    @Override
    public void handleMutation(Mutation m4, int size, int entryLocation, CommitLogDescriptor desc) {
        if (DatabaseDescriptor.isCDCEnabled() && m4.trackedByCDC()) {
            this.sawCDCMutation = true;
        }
        this.pendingMutationBytes += (long)size;
        this.futures.offer(mutationInitiator.initiateMutation(m4, desc.id, size, entryLocation, this));
        while (this.futures.size() > MAX_OUTSTANDING_REPLAY_COUNT || this.pendingMutationBytes > MAX_OUTSTANDING_REPLAY_BYTES || !this.futures.isEmpty() && this.futures.peek().isDone()) {
            this.pendingMutationBytes -= (long)((Integer)FBUtilities.waitOnFuture((java.util.concurrent.Future)this.futures.poll())).intValue();
        }
    }

    @Override
    public boolean shouldSkipSegmentOnError(CommitLogReadHandler.CommitLogReadException exception) throws IOException {
        if (exception.permissible) {
            logger.error("Ignoring commit log replay error likely due to incomplete flush to disk", (Throwable)exception);
        } else if (CassandraRelevantProperties.COMMITLOG_IGNORE_REPLAY_ERRORS.getBoolean()) {
            logger.error("Ignoring commit log replay error", (Throwable)exception);
        } else if (!CommitLog.handleCommitError("Failed commit log replay", exception)) {
            logger.error("Replay stopped. If you wish to override this error and continue starting the node ignoring commit log replay problems, specify -D{}=true on the command line", (Object)CassandraRelevantProperties.COMMITLOG_IGNORE_REPLAY_ERRORS.getKey());
            throw new CommitLogReplayException(exception.getMessage(), exception);
        }
        return false;
    }

    @Override
    public void handleUnrecoverableError(CommitLogReadHandler.CommitLogReadException exception) throws IOException {
        this.shouldSkipSegmentOnError(exception);
    }

    public static class CommitLogReplayException
    extends IOException {
        public CommitLogReplayException(String message, Throwable cause) {
            super(message, cause);
        }

        public CommitLogReplayException(String message) {
            super(message);
        }
    }

    private static class CustomReplayFilter
    extends ReplayFilter {
        private Multimap<String, String> toReplay;

        public CustomReplayFilter(Multimap<String, String> toReplay) {
            this.toReplay = toReplay;
        }

        @Override
        public Iterable<PartitionUpdate> filter(Mutation mutation) {
            final Collection<String> cfNames = this.toReplay.get(mutation.getKeyspaceName());
            if (cfNames == null) {
                return Collections.emptySet();
            }
            return Iterables.filter(mutation.getPartitionUpdates(), new Predicate<PartitionUpdate>(){

                @Override
                public boolean apply(PartitionUpdate upd) {
                    return cfNames.contains(upd.metadata().name);
                }
            });
        }

        @Override
        public boolean includes(TableMetadataRef metadata) {
            return this.toReplay.containsEntry(metadata.keyspace, metadata.name);
        }
    }

    private static class AlwaysReplayFilter
    extends ReplayFilter {
        private AlwaysReplayFilter() {
        }

        @Override
        public Iterable<PartitionUpdate> filter(Mutation mutation) {
            return mutation.getPartitionUpdates();
        }

        @Override
        public boolean includes(TableMetadataRef metadata) {
            return true;
        }
    }

    static abstract class ReplayFilter {
        ReplayFilter() {
        }

        public abstract Iterable<PartitionUpdate> filter(Mutation var1);

        public abstract boolean includes(TableMetadataRef var1);

        public static ReplayFilter create() {
            String replayList = CassandraRelevantProperties.COMMIT_LOG_REPLAY_LIST.getString();
            if (replayList == null) {
                return new AlwaysReplayFilter();
            }
            HashMultimap<String, String> toReplay = HashMultimap.create();
            for (String rawPair : replayList.split(",")) {
                String trimmedRawPair = rawPair.trim();
                if (trimmedRawPair.isEmpty() || trimmedRawPair.endsWith(".")) {
                    throw new IllegalArgumentException(String.format("Invalid pair: '%s'", trimmedRawPair));
                }
                CharSequence[] pair = StringUtils.split((String)trimmedRawPair, (char)'.');
                if (pair.length > 2) {
                    throw new IllegalArgumentException(String.format("%s property contains an item which is not in format 'keyspace' or 'keyspace.table' but it is '%s'", CassandraRelevantProperties.COMMIT_LOG_REPLAY_LIST.getKey(), String.join((CharSequence)".", pair)));
                }
                String keyspaceName = pair[0];
                Keyspace ks = Schema.instance.getKeyspaceInstance(keyspaceName);
                if (ks == null) {
                    throw new IllegalArgumentException("Unknown keyspace " + keyspaceName);
                }
                if (pair.length == 1) {
                    for (ColumnFamilyStore cfs : ks.getColumnFamilyStores()) {
                        toReplay.put(keyspaceName, cfs.name);
                    }
                    continue;
                }
                ColumnFamilyStore cfs = ks.getColumnFamilyStore((String)pair[1]);
                if (cfs == null) {
                    throw new IllegalArgumentException(String.format("Unknown table %s.%s", keyspaceName, pair[1]));
                }
                toReplay.put(keyspaceName, (String)pair[1]);
            }
            if (toReplay.isEmpty()) {
                logger.info("All tables will be included in commit log replay.");
            } else {
                logger.info("Tables to be replayed: {}", (Object)toReplay.asMap().toString());
            }
            return new CustomReplayFilter(toReplay);
        }
    }

    @VisibleForTesting
    public static class MutationInitiator {
        protected Future<Integer> initiateMutation(final Mutation mutation, final long segmentId, int serializedSize, final int entryLocation, final CommitLogReplayer commitLogReplayer) {
            WrappedRunnable runnable = new WrappedRunnable(){

                @Override
                public void runMayThrow() {
                    if (Schema.instance.getKeyspaceMetadata(mutation.getKeyspaceName()) == null) {
                        return;
                    }
                    if (commitLogReplayer.pointInTimeExceeded(mutation)) {
                        return;
                    }
                    Keyspace keyspace = Keyspace.open(mutation.getKeyspaceName());
                    Mutation.PartitionUpdateCollector newPUCollector = null;
                    for (PartitionUpdate update : commitLogReplayer.replayFilter.filter(mutation)) {
                        if (Schema.instance.getTableMetadata(update.metadata().id) == null || !commitLogReplayer.shouldReplay(update.metadata().id, new CommitLogPosition(segmentId, entryLocation))) continue;
                        if (newPUCollector == null) {
                            newPUCollector = new Mutation.PartitionUpdateCollector(mutation.getKeyspaceName(), mutation.key());
                        }
                        newPUCollector.add(update);
                        commitLogReplayer.replayedCount.incrementAndGet();
                    }
                    if (newPUCollector != null) {
                        assert (!newPUCollector.isEmpty());
                        Keyspace.open(newPUCollector.getKeyspaceName()).apply(newPUCollector.build(), false, true, false);
                        commitLogReplayer.keyspacesReplayed.add(keyspace);
                    }
                }
            };
            return Stage.MUTATION.submit(runnable, serializedSize);
        }
    }
}

