/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.streaming;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.Weigher;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.RateLimiter;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanNotificationInfo;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.openmbean.CompositeData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.DurationSpec;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.streaming.SessionInfo;
import org.apache.cassandra.streaming.StreamManagerMBean;
import org.apache.cassandra.streaming.StreamResultFuture;
import org.apache.cassandra.streaming.StreamSession;
import org.apache.cassandra.streaming.StreamingDataOutputPlus;
import org.apache.cassandra.streaming.StreamingState;
import org.apache.cassandra.streaming.management.StreamEventJMXNotifier;
import org.apache.cassandra.streaming.management.StreamStateCompositeData;
import org.apache.cassandra.utils.TimeUUID;
import org.cliffc.high_scale_lib.NonBlockingHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StreamManager
implements StreamManagerMBean {
    private static final Logger logger = LoggerFactory.getLogger(StreamManager.class);
    public static final StreamManager instance = new StreamManager();
    private final StreamEventJMXNotifier notifier = new StreamEventJMXNotifier();
    private final CopyOnWriteArrayList<StreamListener> listeners = new CopyOnWriteArrayList();
    private final Map<TimeUUID, StreamResultFuture> initiatorStreams = new NonBlockingHashMap<TimeUUID, StreamResultFuture>();
    private final Map<TimeUUID, StreamResultFuture> followerStreams = new NonBlockingHashMap<TimeUUID, StreamResultFuture>();
    private final Cache<TimeUUID, StreamingState> states;
    private final StreamListener listener = new StreamListener(){

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void onRegister(StreamResultFuture result) {
            if (!DatabaseDescriptor.getStreamingStatsEnabled()) {
                return;
            }
            Cache<TimeUUID, StreamingState> cache = StreamManager.this.states;
            synchronized (cache) {
                StreamingState previous = StreamManager.this.states.getIfPresent(result.planId);
                if (previous == null) {
                    StreamingState state = new StreamingState(result);
                    StreamManager.this.states.put(state.id(), state);
                    state.phase.start();
                    result.addEventListener(state);
                } else {
                    logger.warn("Duplicate streaming states detected for id {}", (Object)result.planId);
                }
            }
        }
    };

    public static StreamRateLimiter getRateLimiter(InetAddressAndPort peer) {
        return new StreamRateLimiter(peer, StreamRateLimiter.LIMITER, StreamRateLimiter.INTER_DC_LIMITER, DatabaseDescriptor.getStreamThroughputOutboundBytesPerSec(), DatabaseDescriptor.getInterDCStreamThroughputOutboundBytesPerSec());
    }

    public static StreamRateLimiter getEntireSSTableRateLimiter(InetAddressAndPort peer) {
        return new StreamRateLimiter(peer, StreamRateLimiter.ENTIRE_SSTABLE_LIMITER, StreamRateLimiter.ENTIRE_SSTABLE_INTER_DC_LIMITER, DatabaseDescriptor.getEntireSSTableStreamThroughputOutboundBytesPerSec(), DatabaseDescriptor.getEntireSSTableInterDCStreamThroughputOutboundBytesPerSec());
    }

    protected void addStreamingStateAgain(StreamingState state) {
        if (!DatabaseDescriptor.getStreamingStatsEnabled()) {
            return;
        }
        this.states.put(state.id(), state);
    }

    public StreamManager() {
        DurationSpec.LongNanosecondsBound duration = DatabaseDescriptor.getStreamingStateExpires();
        long sizeBytes = DatabaseDescriptor.getStreamingStateSize().toBytes();
        logger.info("Storing streaming state for {} or for size {}", (Object)duration, (Object)sizeBytes);
        this.states = CacheBuilder.newBuilder().expireAfterWrite(duration.quantity(), duration.unit()).maximumWeight(sizeBytes).weigher(new StreamingStateWeigher()).build();
    }

    public void start() {
        this.addListener(this.listener);
    }

    public void stop() {
        this.removeListener(this.listener);
    }

    public Collection<StreamingState> getStreamingStates() {
        return this.states.asMap().values();
    }

    public StreamingState getStreamingState(TimeUUID id) {
        return this.states.getIfPresent(id);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public void putStreamingState(StreamingState state) {
        Cache<TimeUUID, StreamingState> cache = this.states;
        synchronized (cache) {
            StreamingState previous = this.states.getIfPresent(state.id());
            if (previous != null) {
                throw new AssertionError((Object)("StreamPlan id " + state.id() + " already exists"));
            }
            this.states.put(state.id(), state);
        }
    }

    @VisibleForTesting
    public void clearStates() {
        this.states.asMap().clear();
    }

    @Override
    public Set<CompositeData> getCurrentStreams() {
        return Sets.newHashSet(Iterables.transform(Iterables.concat(this.initiatorStreams.values(), this.followerStreams.values()), new Function<StreamResultFuture, CompositeData>(){

            @Override
            public CompositeData apply(StreamResultFuture input) {
                return StreamStateCompositeData.toCompositeData(input.getCurrentState());
            }
        }));
    }

    @Override
    public boolean getStreamingStatsEnabled() {
        return DatabaseDescriptor.getStreamingStatsEnabled();
    }

    @Override
    public void setStreamingStatsEnabled(boolean streamingStatsEnabled) {
        DatabaseDescriptor.setStreamingStatsEnabled(streamingStatsEnabled);
    }

    @Override
    public String getStreamingSlowEventsLogTimeout() {
        return DatabaseDescriptor.getStreamingSlowEventsLogTimeout().toString();
    }

    @Override
    public void setStreamingSlowEventsLogTimeout(String value) {
        DatabaseDescriptor.setStreamingSlowEventsLogTimeout(value);
    }

    public void registerInitiator(StreamResultFuture result) {
        result.addEventListener(this.notifier);
        result.addListener(() -> this.initiatorStreams.remove(result.planId));
        this.initiatorStreams.put(result.planId, result);
        this.notifySafeOnRegister(result);
    }

    public StreamResultFuture registerFollower(StreamResultFuture result) {
        result.addEventListener(this.notifier);
        result.addListener(() -> this.followerStreams.remove(result.planId));
        StreamResultFuture previous = this.followerStreams.putIfAbsent(result.planId, result);
        if (previous == null) {
            this.notifySafeOnRegister(result);
            return result;
        }
        return previous;
    }

    @VisibleForTesting
    public void putInitiatorStream(StreamResultFuture future) {
        StreamResultFuture current = this.initiatorStreams.putIfAbsent(future.planId, future);
        assert (current == null) : "Duplicat initiator stream for " + future.planId;
    }

    @VisibleForTesting
    public void putFollowerStream(StreamResultFuture future) {
        StreamResultFuture current = this.followerStreams.putIfAbsent(future.planId, future);
        assert (current == null) : "Duplicate follower stream for " + future.planId;
    }

    public void addListener(StreamListener listener) {
        this.listeners.add(listener);
    }

    public void removeListener(StreamListener listener) {
        this.listeners.remove(listener);
    }

    private void notifySafeOnRegister(StreamResultFuture result) {
        for (StreamListener l : this.listeners) {
            try {
                l.onRegister(result);
            }
            catch (Throwable t2) {
                logger.warn("Failed to notify stream listener of new Initiator/Follower", t2);
            }
        }
    }

    public StreamResultFuture getReceivingStream(TimeUUID planId) {
        return this.followerStreams.get(planId);
    }

    public StreamResultFuture getInitiatorStream(TimeUUID planId) {
        return this.initiatorStreams.get(planId);
    }

    @Override
    public void addNotificationListener(NotificationListener listener, NotificationFilter filter, Object handback) {
        this.notifier.addNotificationListener(listener, filter, handback);
    }

    @Override
    public void removeNotificationListener(NotificationListener listener) throws ListenerNotFoundException {
        this.notifier.removeNotificationListener(listener);
    }

    @Override
    public void removeNotificationListener(NotificationListener listener, NotificationFilter filter, Object handback) throws ListenerNotFoundException {
        this.notifier.removeNotificationListener(listener, filter, handback);
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return this.notifier.getNotificationInfo();
    }

    public StreamSession findSession(InetAddressAndPort peer, TimeUUID planId, int sessionIndex, boolean searchInitiatorSessions) {
        Map<TimeUUID, StreamResultFuture> streams = searchInitiatorSessions ? this.initiatorStreams : this.followerStreams;
        return this.findSession(streams, peer, planId, sessionIndex);
    }

    private StreamSession findSession(Map<TimeUUID, StreamResultFuture> streams, InetAddressAndPort peer, TimeUUID planId, int sessionIndex) {
        StreamResultFuture streamResultFuture = streams.get(planId);
        if (streamResultFuture == null) {
            return null;
        }
        return streamResultFuture.getSession(peer, sessionIndex);
    }

    public long getTotalRemainingOngoingBytes() {
        long total = 0L;
        for (StreamResultFuture fut : Iterables.concat(this.initiatorStreams.values(), this.followerStreams.values())) {
            for (SessionInfo sessionInfo : fut.getCurrentState().sessions) {
                total += sessionInfo.getTotalSizeToReceive() - sessionInfo.getTotalSizeReceived();
            }
        }
        return total;
    }

    public static interface StreamListener {
        default public void onRegister(StreamResultFuture result) {
        }
    }

    private static class StreamingStateWeigher
    implements Weigher<TimeUUID, StreamingState> {
        private StreamingStateWeigher() {
        }

        @Override
        public int weigh(TimeUUID key, StreamingState val) {
            long costOfStreamingState = val.unsharedHeapSize() + TimeUUID.TIMEUUID_SIZE;
            return Math.toIntExact(costOfStreamingState);
        }
    }

    public static class StreamRateLimiter
    implements StreamingDataOutputPlus.RateLimiter {
        public static final double BYTES_PER_MEBIBYTE = 1048576.0;
        private static final RateLimiter LIMITER = RateLimiter.create(StreamRateLimiter.calculateRateInBytes());
        private static final RateLimiter INTER_DC_LIMITER = RateLimiter.create(StreamRateLimiter.calculateInterDCRateInBytes());
        private static final RateLimiter ENTIRE_SSTABLE_LIMITER = RateLimiter.create(StreamRateLimiter.calculateEntireSSTableRateInBytes());
        private static final RateLimiter ENTIRE_SSTABLE_INTER_DC_LIMITER = RateLimiter.create(StreamRateLimiter.calculateEntireSSTableInterDCRateInBytes());
        private final RateLimiter limiter;
        private final RateLimiter interDCLimiter;
        private final boolean isLocalDC;
        private final double throughput;
        private final double interDCThroughput;

        private StreamRateLimiter(InetAddressAndPort peer, RateLimiter limiter, RateLimiter interDCLimiter, double throughput, double interDCThroughput) {
            this.limiter = limiter;
            this.interDCLimiter = interDCLimiter;
            this.throughput = throughput;
            this.interDCThroughput = interDCThroughput;
            this.isLocalDC = DatabaseDescriptor.getLocalDataCenter() != null && DatabaseDescriptor.getEndpointSnitch() != null ? DatabaseDescriptor.getLocalDataCenter().equals(DatabaseDescriptor.getEndpointSnitch().getDatacenter(peer)) : true;
        }

        @Override
        public void acquire(int toTransfer) {
            this.limiter.acquire(toTransfer);
            if (!this.isLocalDC) {
                this.interDCLimiter.acquire(toTransfer);
            }
        }

        @Override
        public boolean isRateLimited() {
            return this.throughput > 0.0 || !this.isLocalDC && this.interDCThroughput > 0.0;
        }

        public static void updateThroughput() {
            LIMITER.setRate(StreamRateLimiter.calculateRateInBytes());
        }

        public static void updateInterDCThroughput() {
            INTER_DC_LIMITER.setRate(StreamRateLimiter.calculateInterDCRateInBytes());
        }

        public static void updateEntireSSTableThroughput() {
            ENTIRE_SSTABLE_LIMITER.setRate(StreamRateLimiter.calculateEntireSSTableRateInBytes());
        }

        public static void updateEntireSSTableInterDCThroughput() {
            ENTIRE_SSTABLE_INTER_DC_LIMITER.setRate(StreamRateLimiter.calculateEntireSSTableInterDCRateInBytes());
        }

        private static double calculateRateInBytes() {
            double throughput = DatabaseDescriptor.getStreamThroughputOutboundBytesPerSec();
            return StreamRateLimiter.calculateEffectiveRateInBytes(throughput);
        }

        private static double calculateInterDCRateInBytes() {
            double throughput = DatabaseDescriptor.getInterDCStreamThroughputOutboundBytesPerSec();
            return StreamRateLimiter.calculateEffectiveRateInBytes(throughput);
        }

        private static double calculateEntireSSTableRateInBytes() {
            double throughput = DatabaseDescriptor.getEntireSSTableStreamThroughputOutboundBytesPerSec();
            return StreamRateLimiter.calculateEffectiveRateInBytes(throughput);
        }

        private static double calculateEntireSSTableInterDCRateInBytes() {
            double throughput = DatabaseDescriptor.getEntireSSTableInterDCStreamThroughputOutboundBytesPerSec();
            return StreamRateLimiter.calculateEffectiveRateInBytes(throughput);
        }

        @VisibleForTesting
        public static double getRateLimiterRateInBytes() {
            return LIMITER.getRate();
        }

        @VisibleForTesting
        public static double getInterDCRateLimiterRateInBytes() {
            return INTER_DC_LIMITER.getRate();
        }

        @VisibleForTesting
        public static double getEntireSSTableRateLimiterRateInBytes() {
            return ENTIRE_SSTABLE_LIMITER.getRate();
        }

        @VisibleForTesting
        public static double getEntireSSTableInterDCRateLimiterRateInBytes() {
            return ENTIRE_SSTABLE_INTER_DC_LIMITER.getRate();
        }

        private static double calculateEffectiveRateInBytes(double throughput) {
            return throughput > 0.0 ? throughput : Double.MAX_VALUE;
        }
    }
}

