/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.dolphinscheduler.plugin.task.api.log;

import org.apache.dolphinscheduler.plugin.task.api.utils.LogUtils;

import org.apache.commons.lang3.StringUtils;

import lombok.extern.slf4j.Slf4j;

import org.slf4j.MDC;
import org.slf4j.Marker;

import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.filter.Filter;
import ch.qos.logback.core.spi.FilterReply;

/**
 * This class is used to filter the log of the task instance.
 */
@Slf4j
public class TaskLogFilter extends Filter<ILoggingEvent> {

    @Override
    public FilterReply decide(ILoggingEvent event) {
        String taskInstanceLogPath = MDC.get(LogUtils.TASK_INSTANCE_LOG_FULL_PATH_MDC_KEY);
        // If the taskInstanceLogPath is empty, it means that the log is not related to a task instance.
        if (StringUtils.isEmpty(taskInstanceLogPath)) {
            return FilterReply.DENY;
        }

        // todo: Only when the master is includeInTaskLog, the log should be included in the task instance log, we can
        // remove `excludeInTaskLog` after change the log in plugin
        final Marker marker = event.getMarker();
        if (marker == null) {
            return FilterReply.ACCEPT;
        }
        if (marker.contains(TaskLogMarkers.includeInTaskLog())) {
            return FilterReply.ACCEPT;
        }
        if (marker.contains(TaskLogMarkers.excludeInTaskLog())) {
            return FilterReply.DENY;
        }
        return FilterReply.ACCEPT;
    }
}
