/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.python.util;

import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.streaming.api.functions.sink.legacy.SinkFunction;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/** A SinkFunction for collecting results of DataStream transformations in test cases. */
public class DataStreamTestCollectSink<IN> implements SinkFunction<IN> {

    private final int id;

    private static final List<Tuple2<Integer, Object>> collectedResult = new ArrayList<>();

    public DataStreamTestCollectSink() {
        id = hashCode();
    }

    /**
     * Collect the sink value into a static List so that the client side can fetch the result of
     * flink job in test cases. if the output value is a byte array generated by pickle, it will be
     * added to the list directly letting the client to deserialize the pickled bytes to python
     * objects. Otherwise, the value will be added to the list in string format.
     */
    @Override
    public void invoke(IN value, Context context) throws Exception {

        synchronized (collectedResult) {
            collectedResult.add(Tuple2.of(id, value));
        }
    }

    public List<Object> collectAndClear(boolean isPythonObjects) {
        List<Object> listToBeReturned;
        synchronized (collectedResult) {
            listToBeReturned =
                    collectedResult.stream()
                            .filter(e -> e.f0.equals(id))
                            .map(e -> e.f1)
                            .collect(Collectors.toList());
        }
        if (!isPythonObjects) {
            for (int i = 0; i < listToBeReturned.size(); i++) {
                listToBeReturned.set(i, listToBeReturned.get(i).toString());
            }
        }
        clear();
        return listToBeReturned;
    }

    public void clear() {
        synchronized (collectedResult) {
            collectedResult.removeIf(e -> e.f0.equals(id));
        }
    }
}
